package com.example.tramitedoc

import android.content.Context
import android.content.Intent
import android.net.Uri
import android.os.Bundle
import android.provider.OpenableColumns
import android.util.Log
import android.widget.Toast
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.result.contract.ActivityResultContracts
import androidx.activity.compose.rememberLauncherForActivityResult
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Info
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewmodel.compose.viewModel
import androidx.navigation.NavController
import androidx.navigation.compose.*
import com.android.volley.Request
import com.android.volley.toolbox.StringRequest
import com.android.volley.toolbox.Volley
import org.json.JSONObject
import java.util.*

/* ------------------------------
   LISTA DE ÁREAS
--------------------------------*/

val listaAreas = listOf(
    "Mesa de Partes",
    "Secretaría Académica",
    "Dirección",
    "Coordinación",
    "Tesorería",
    "Archivo Central"
)

/* ------------------------------
   DATA CLASS
--------------------------------*/

data class Tramite(
    val id: String = UUID.randomUUID().toString(),
    val dni: String,
    val nombres: String,
    val apellidos: String,
    val celular: String,
    val tipoDocumento: String,
    val descripcion: String,
    val pdfUri: String?,
    val fecha: String,
    val estado: String = "Registrado",
    val areaDestino: String,
    val observaciones: String = "Sin observaciones"
)

/* ------------------------------
   VIEWMODEL
--------------------------------*/

class TramiteViewModel : ViewModel() {

    var listaTramites = mutableStateListOf<Tramite>()
        private set

    fun registrarTramiteLocal(
        dni: String,
        nombres: String,
        apellidos: String,
        celular: String,
        tipoDocumento: String,
        descripcion: String,
        area: String,
        pdfUri: String?
    ) {
        val nuevo = Tramite(
            dni = dni,
            nombres = nombres,
            apellidos = apellidos,
            celular = celular,
            tipoDocumento = tipoDocumento,
            descripcion = descripcion,
            pdfUri = pdfUri,
            fecha = Date().toString(),
            areaDestino = area
        )
        listaTramites.add(0, nuevo)
    }

    fun obtenerTramite(id: String): Tramite? {
        return listaTramites.find { it.id == id }
    }

    fun enviarTramiteAlServidor(
        context: Context,
        tramite: Tramite,
        urlServidor: String,
        onResult: (Boolean, String?) -> Unit
    ) {

        val request = object : StringRequest(
            Method.POST,
            urlServidor,
            { onResult(true, it) },
            { onResult(false, it.message ?: "Error") }
        ) {
            override fun getParams(): MutableMap<String, String> {
                return hashMapOf(
                    "dni" to tramite.dni,
                    "nombres" to tramite.nombres,
                    "apellidos" to tramite.apellidos,
                    "celular" to tramite.celular,
                    "tipo_documento" to tramite.tipoDocumento,
                    "descripcion" to tramite.descripcion,
                    "area_destino" to tramite.areaDestino,
                    "observaciones" to tramite.observaciones,
                    "pdf_url" to (tramite.pdfUri ?: "")
                )
            }
        }

        Volley.newRequestQueue(context).add(request)
    }

    fun cargarTramitesDesdeServidor(context: Context, url: String) {

        val request = StringRequest(
            Request.Method.GET,
            url,
            { response ->

                try {

                    Log.d("RESPUESTA_SERVER", response)

                    if (response.isNullOrBlank()) {
                        Log.e("ERROR_LISTA", "Respuesta vacía del servidor")
                        return@StringRequest
                    }

                    val json = JSONObject(response)
                    val lista = json.getJSONArray("data")
                    listaTramites.clear()

                    for (i in 0 until lista.length()) {
                        val o = lista.getJSONObject(i)
                        listaTramites.add(
                            Tramite(
                                id = o.optString("id"),
                                dni = o.optString("dni"),
                                nombres = o.optString("nombres"),
                                apellidos = o.optString("apellidos"),
                                celular = o.optString("celular"),
                                tipoDocumento = o.optString("tipo_documento"),
                                descripcion = o.optString("descripcion"),
                                pdfUri = o.optString("pdf_url"),
                                fecha = o.optString("fecha"),
                                estado = o.optString("estado"),
                                areaDestino = o.optString("area_destino"),
                                observaciones = o.optString("observaciones")
                            )
                        )
                    }

                } catch (e: Exception) {
                    Log.e("ERROR_JSON", e.toString())
                }
            },
            { error ->
                Log.e("ERROR_VOLLEY", error.toString())
            }
        )

        Volley.newRequestQueue(context).add(request)
    }



    fun consultarSeguimientoServidor(
        context: Context,
        codigo: String,
        url: String,
        onResult: (Tramite?) -> Unit
    ) {

        val request = object : StringRequest(
            Method.POST,
            url,
            { response ->
                try {
                    val obj = JSONObject(response)
                    if (obj.getString("status") == "success") {

                        val data = obj.getJSONObject("data")

                        val t = Tramite(
                            id = data.getString("id"),
                            dni = data.getString("dni"),
                            nombres = data.getString("nombres"),
                            apellidos = data.getString("apellidos"),
                            celular = data.getString("celular"),
                            tipoDocumento = data.getString("tipo_documento"),
                            descripcion = data.getString("descripcion"),
                            pdfUri = data.optString("pdf_url"),
                            fecha = data.getString("fecha"),
                            estado = data.getString("estado"),
                            areaDestino = data.getString("area_destino"),
                            observaciones = data.optString("observaciones", "Sin observaciones")
                        )

                        onResult(t)
                    } else {
                        onResult(null)
                    }

                } catch (e: Exception) {
                    onResult(null)
                }
            },
            { onResult(null) }
        ) {
            override fun getParams(): MutableMap<String, String> {
                return hashMapOf("codigo" to codigo)
            }
        }

        Volley.newRequestQueue(context).add(request)
    }
}

/* ------------------------------
   MAIN
--------------------------------*/

class MainActivity : ComponentActivity() {
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContent {
            MaterialTheme {
                PantallaNavegacion()
            }
        }
    }
}

/* ------------------------------
   TARJETA
--------------------------------*/

@Composable
fun TarjetaTramite(t: Tramite, onClick: () -> Unit) {

    Card(
        modifier = Modifier
            .fillMaxWidth()
            .padding(10.dp)
            .clickable { onClick() }
    ) {
        Column(modifier = Modifier.padding(12.dp)) {
            Text("DNI: ${t.dni}")
            Text("Nombre: ${t.nombres} ${t.apellidos}")
            Text("Área: ${t.areaDestino}")
            Text("Estado: ${t.estado}")
        }
    }
}

/* ------------------------------
   REGISTRO
--------------------------------*/

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun PantallaRegistrar(vm: TramiteViewModel) {

    val ctx = LocalContext.current

    var dni by remember { mutableStateOf("") }
    var nombres by remember { mutableStateOf("") }
    var apellidos by remember { mutableStateOf("") }
    var celular by remember { mutableStateOf("") }
    var tipo by remember { mutableStateOf("") }
    var desc by remember { mutableStateOf("") }

    var area by remember { mutableStateOf(listaAreas[0]) }
    var expanded by remember { mutableStateOf(false) }

    var pdfUri by remember { mutableStateOf<Uri?>(null) }
    var pdfNombre by remember { mutableStateOf("") }

    val pickPDF = rememberLauncherForActivityResult(ActivityResultContracts.GetContent()) {
        pdfUri = it
        it?.let { uri ->
            val cursor = ctx.contentResolver.query(uri, null, null, null, null)
            cursor?.moveToFirst()
            pdfNombre = cursor?.getString(cursor.getColumnIndex(OpenableColumns.DISPLAY_NAME)) ?: ""
            cursor?.close()
        }
    }

    Column(
        modifier = Modifier
            .padding(16.dp)
            .verticalScroll(rememberScrollState())
    ) {

        Text("SISTEMA DE TRÁMITE DOCUMENTARIO", fontSize = 18.sp)
        Spacer(Modifier.height(4.dp))
        Text("Registrar Trámite", fontSize = 14.sp)

        Spacer(Modifier.height(10.dp))

        OutlinedTextField(dni, { dni = it }, label = { Text("DNI") }, modifier = Modifier.fillMaxWidth())
        OutlinedTextField(nombres, { nombres = it }, label = { Text("Nombres") }, modifier = Modifier.fillMaxWidth())
        OutlinedTextField(apellidos, { apellidos = it }, label = { Text("Apellidos") }, modifier = Modifier.fillMaxWidth())
        OutlinedTextField(celular, { celular = it }, label = { Text("Celular") }, modifier = Modifier.fillMaxWidth())
        OutlinedTextField(tipo, { tipo = it }, label = { Text("Documento") }, modifier = Modifier.fillMaxWidth())
        OutlinedTextField(desc, { desc = it }, label = { Text("Descripción") }, modifier = Modifier.fillMaxWidth())

        Spacer(Modifier.height(8.dp))

        ExposedDropdownMenuBox(expanded = expanded, onExpandedChange = { expanded = !expanded }) {

            OutlinedTextField(
                value = area,
                onValueChange = {},
                readOnly = true,
                label = { Text("Área de destino") },
                trailingIcon = { ExposedDropdownMenuDefaults.TrailingIcon(expanded) },
                modifier = Modifier
                    .menuAnchor()
                    .fillMaxWidth()
            )

            ExposedDropdownMenu(expanded = expanded, onDismissRequest = { expanded = false }) {
                listaAreas.forEach { opcion ->
                    DropdownMenuItem(
                        text = { Text(opcion) },
                        onClick = {
                            area = opcion
                            expanded = false
                        }
                    )
                }
            }
        }

        Spacer(Modifier.height(10.dp))

        Button(onClick = { pickPDF.launch("application/pdf") }) {
            Text("Seleccionar PDF")
        }

        Text(pdfNombre)

        Spacer(Modifier.height(10.dp))

        Button(onClick = {

            vm.registrarTramiteLocal(
                dni, nombres, apellidos, celular, tipo, desc, area, pdfUri?.toString()
            )

            val t = Tramite(
                dni = dni,
                nombres = nombres,
                apellidos = apellidos,
                celular = celular,
                tipoDocumento = tipo,
                descripcion = desc,
                areaDestino = area,
                pdfUri = pdfUri?.toString(),
                fecha = Date().toString()
            )

            vm.enviarTramiteAlServidor(
                ctx,
                t,
                "http://cristhian.grupodvh.com/tramiteapi/registrar_tramite.php"
            ) { ok, err ->
                Toast.makeText(
                    ctx,
                    if (ok) "Trámite registrado correctamente" else err ?: "Error",
                    Toast.LENGTH_SHORT
                ).show()
            }

        }) {
            Text("GUARDAR TRÁMITE")
        }
    }
}

/* ------------------------------
   LISTADO + BOTÓN PDF REPORTE
--------------------------------*/

@Composable
fun PantallaListado(nav: NavController, vm: TramiteViewModel) {

    val ctx = LocalContext.current

    LaunchedEffect(Unit) {
        vm.cargarTramitesDesdeServidor(
            ctx,
            "http://cristhian.grupodvh.com/tramiteapi/listar_tramites.php"
        )
    }

    Column(Modifier.fillMaxSize()) {

        // BOTÓN PARA GENERAR REPORTE PDF
        Button(
            onClick = {
                val intent = Intent(
                    Intent.ACTION_VIEW,
                    Uri.parse("http://cristhian.grupodvh.com/tramiteapi/reporte_tramites.php")
                )
                ctx.startActivity(intent)
            },
            modifier = Modifier
                .padding(12.dp)
                .fillMaxWidth()
        ) {
            Text("📄 Generar Reporte PDF")
        }

        // BOTÓN SEGUIMIENTO
        Button(
            onClick = { nav.navigate("seguimiento") },
            modifier = Modifier
                .padding(horizontal = 12.dp)
                .fillMaxWidth()
        ) {
            Text("🔎 Seguimiento")
        }

        Spacer(Modifier.height(10.dp))

        vm.listaTramites.forEach {
            TarjetaTramite(it) {
                nav.navigate("detalle/${it.id}")
            }
        }
    }
}

/* ------------------------------
   SEGUIMIENTO
--------------------------------*/

@Composable
fun PantallaSeguimiento(vm: TramiteViewModel) {

    val ctx = LocalContext.current
    var codigo by remember { mutableStateOf("") }
    var resultado by remember { mutableStateOf<Tramite?>(null) }
    var cargando by remember { mutableStateOf(false) }

    Column(Modifier.padding(16.dp)) {

        Text("Seguimiento de Trámite", fontSize = 22.sp)

        OutlinedTextField(
            value = codigo,
            onValueChange = { codigo = it },
            label = { Text("DNI o código") },
            modifier = Modifier.fillMaxWidth()
        )

        Spacer(Modifier.height(6.dp))

        Button(onClick = {
            cargando = true
            resultado = null

            vm.consultarSeguimientoServidor(
                ctx,
                codigo,
                "http://cristhian.grupodvh.com/tramiteapi/seguimiento.php"
            ) {
                cargando = false
                resultado = it
            }

        }) {
            Text("BUSCAR")
        }

        Spacer(Modifier.height(10.dp))

        if (resultado != null) {

            Text("Estado: ${resultado!!.estado}")
            Text("Área: ${resultado!!.areaDestino}")
            Text("Observaciones: ${resultado!!.observaciones}")

        } else if (!cargando && codigo.isNotEmpty()) {

            Text("No se encontró el trámite.")

        }
    }
}

/* ------------------------------
   DETALLE
--------------------------------*/

@Composable
fun PantallaDetalle(id: String, vm: TramiteViewModel) {

    val t = vm.obtenerTramite(id) ?: return

    Column(Modifier.padding(16.dp)) {

        Text("Detalle del trámite", fontSize = 20.sp)
        Spacer(Modifier.height(8.dp))

        Text("DNI: ${t.dni}")
        Text("Nombre: ${t.nombres} ${t.apellidos}")
        Text("Área: ${t.areaDestino}")
        Text("Estado: ${t.estado}")
        Text("Observaciones: ${t.observaciones}")
    }
}

/* ------------------------------
   INFO
--------------------------------*/

@Composable
fun PantallaInfo() {
    Column(Modifier.padding(16.dp)) {
        Text("Sistema de Trámite Documentario\nVersión 1.0\nJetpack Compose")
    }
}

/* ------------------------------
   NAVEGACIÓN
--------------------------------*/

@Composable
fun PantallaNavegacion() {

    val nav = rememberNavController()
    val vm: TramiteViewModel = viewModel()

    Scaffold(
        bottomBar = {
            NavigationBar {
                listOf("registrar", "listado", "info").forEach { ruta ->
                    NavigationBarItem(
                        selected = false,
                        icon = { Icon(Icons.Default.Info, "") },
                        label = { Text(ruta) },
                        onClick = { nav.navigate(ruta) }
                    )
                }
            }
        }
    ) { padding ->

        NavHost(nav, startDestination = "registrar", Modifier.padding(padding)) {
            composable("registrar") { PantallaRegistrar(vm) }
            composable("listado") { PantallaListado(nav, vm) }
            composable("seguimiento") { PantallaSeguimiento(vm) }
            composable("detalle/{id}") {
                PantallaDetalle(it.arguments?.getString("id") ?: "", vm)
            }
            composable("info") { PantallaInfo() }
        }
    }
}
